<?php
namespace WPC;

/**
 * Validate the user-selected components against the available components.
 *
 * @param array $components The available components.
 * @param array $user_selected_components The components selected by the user.
 * @return array List of error messages, if any.
 */
function configurator_validate_components( $components, $user_selected_components ) {
	$errors = array();

	foreach ( $components as $component ) {
		$component_name = $component['name'];
		$children       = $component['children'] ?? array();
		$required       = $component['settings']['required'] ?? false;

		if ( $component['uid'] !== 'Qeh0-rF3y' ) {
			// return $errors;
		}

		if ( $required ) {
			if ( empty( $children ) ) {
				$errors[ $component['uid'] ] = sprintf( __( "No options found for the group '%s'." ), $component_name );
			}

			// Check if the user has selected any components for this component
			if ( empty( $user_selected_components[ $component['uid'] ] ) ) {
				$errors[ $component['uid'] ] = sprintf( __( "You did not select any options for '%s'." ), $component_name );

				continue; // Skip further validation for this component if no options are selected
			}

			$errors = array_merge( $errors, configurator_validate_group( $component, $user_selected_components[ $component['uid'] ]['active'] ) );
		}
	}

	return $errors;
}

/**
 * Validate a component and its children.
 *
 * @param array $component The component being validated.
 * @param array $user_selected_components The components selected by the user.
 * @return array List of error messages, if any.
 */
function configurator_validate_group( $component, $user_selected_components = array() ) {
	if ( empty( $user_selected_components ) ) {
		return array( $component['uid'] => sprintf( __( "You did not select any options for '%s'." ), $component['name'] ) );
	}

	$total_subgroups = configurator_count_subgroups( $component['children'] );
	$total_images    = configurator_count_images( $component['children'] );

	// If has nested subgroups, validate them.
	if ( $total_subgroups > 0 ) {
		$subgroup_errors = array();

		foreach ( $component['children'] as $value ) {
			if ( 'sub_group' === $value['type'] ) {
				$subgroup_selected_components = configurator_get_subgroup_selected_components( $value['uid'], $user_selected_components );
				$subgroup_errors              = array_merge( $subgroup_errors, configurator_validate_subgroup( $value, $subgroup_selected_components, $component['name'] ) );
			}
		}

		return $subgroup_errors; // Return early if there are subgroup errors.
	} elseif ( $total_images > 0 ) {
		// If no subgroups but images are present, validate images.
		return configurator_validate_required_image_children( $component, $user_selected_components );
	}
}

/**
 * Validate the subgroup against the selected components.
 *
 * @param array  $subgroup The subgroup being validated.
 * @param array  $user_selected_components The components selected by the user.
 * @param string $component_name The name of the component for error messages.
 * @return array List of error messages, if any.
 */
function configurator_validate_subgroup( $subgroup, $user_selected_components, $component_name = '' ) {
	$errors = array();

	$name     = $subgroup['name'] ?? '';
	$required = $subgroup['settings']['required'] ?? false;
	$children = $subgroup['children'] ?? array();

	// error_log( 'Validating subgroup: ' . $name );
	// error_log( print_r( $user_selected_components, true ) ); // Debugging line

	if ( $required ) {
		if ( empty( $children ) ) {
			$errors[ $subgroup['uid'] ] = sprintf( "No options found for subgroup '%1\$s' in group '%2\$s'.", $name, $component_name );
			return $errors; // Early return if no children are found
		}

		if ( empty( $user_selected_components ) ) {
			$errors[ $subgroup['uid'] ] = sprintf( "You did not select any options for subgroup '%1\$s' in group '%2\$s'.", $name, $component_name );
			return $errors; // Early return if no user-selected components are found
		}

		$total_subgroups = configurator_count_subgroups( $children );

		if ( $total_subgroups > 0 ) {
			// Validate subgroups recursively
			foreach ( $children as $child ) {
				if ( 'sub_group' === $child['type'] ) {
					$subgroup_selected_components = configurator_get_subgroup_selected_components( $child['uid'], $user_selected_components );
					$errors                       = array_merge( $errors, configurator_validate_subgroup( $child, $subgroup_selected_components, $component_name ) );
				}
			}

			if ( ! empty( $errors ) ) {
				return $errors; // Return early if there are errors in subgroups
			}
		} else {
			// Get all child component UIDs
			$children_uids                = wp_list_pluck( $children, 'uid' );
			$subgroup_selected_components = configurator_get_subgroup_selected_components( $subgroup['uid'], $user_selected_components );

			// Get subgroup and image IDs from the user-selected components
			$subgroup_and_image_ids = configurator_get_subgroup_and_image_ids( $subgroup['uid'], $user_selected_components );

			// Check if the selected subgroup and image IDs are valid
			if ( $subgroup['uid'] !== $subgroup_and_image_ids['sub_group'] || ! in_array( $subgroup_and_image_ids['image'], $children_uids ) ) {
				$errors[ $subgroup['uid'] ] = sprintf( "Please select a valid option for subgroup '%1\$s' in group '%2\$s'.", $name, $component_name );
				return $errors; // Early return if the selected subgroup or image is invalid
			}
		}
	}

	return $errors;
}

/**
 * Get subgroup and image IDs from a list of items.
 *
 * @param string $subgroup_uid The UID of the subgroup to search for.
 * @param array  $items List of items containing 'uid' and 'type' keys.
 * @return array Associative array with 'sub_group' and 'image' IDs or null if not found.
 */
function configurator_get_subgroup_and_image_ids( $subgroup_uid, $items ) {
	$result = array(
		'sub_group' => null,
		'image'     => null,
	);

	foreach ( $items as $item ) {
		if ( ! isset( $item['type'], $item['uid'] ) ) {
			continue; // Skip if 'type' or 'uid' is missing.
		}

		if ( 'sub_group' === $item['type'] && $item['uid'] === $subgroup_uid ) {
			$result['sub_group'] = $item['uid'];
		} elseif ( 'image' === $item['type'] ) {
			$result['image'] = $item['uid'];
		}

		// Stop early if both are found.
		if ( $result['sub_group'] && $result['image'] ) {
			break;
		}
	}

	return $result;
}


/**
 * Count the total number of items of a specific type in the array.
 *
 * @param array  $array The input array to check.
 * @param string $type The type of item to count (e.g., 'sub_group', 'image').
 * @return int The total number of items of the specified type.
 */
function configurator_count_items_by_type( $array, $type ) {
	if ( ! is_array( $array ) || empty( $type ) ) {
		return 0;
	}

	$item_count = 0;

	foreach ( $array as $item ) {
		if ( isset( $item['type'] ) && $item['type'] === $type ) {
			++$item_count;
		}
	}

	return $item_count;
}

/**
 * Count the total number of subgroups in the array.
 *
 * @param array $array The input array to check.
 * @return int The total number of subgroups.
 */
function configurator_count_subgroups( $array ) {
	return configurator_count_items_by_type( $array, 'sub_group' );
}

/**
 * Count the total number of images in the array.
 *
 * @param array $array The input array to check.
 * @return int The total number of images.
 */
function configurator_count_images( $array ) {
	return configurator_count_items_by_type( $array, 'image' );
}

/**
 * Extracts all 'uid' values from a nested array.
 *
 * @param array $array The array to search through.
 * @return array An array of 'uid' values found in the input array.
 */
function configurator_get_uids_from_array( $array ) {
	$uids = array();

	// Recursively search for 'uid' in the array
	array_walk_recursive(
		$array,
		function ( $value, $key ) use ( &$uids ) {
			if ( $key === 'uid' ) {
				$uids[] = $value; // Add the 'uid' value to the result array
			}
		}
	);

	return $uids;
}

/** * Retrieves the selected components for a specific subgroup.
 *
 * @param string $subgroup_uid The UID of the subgroup to search for.
 * @param array  $active_components The active components from which to retrieve the selected components.
 * @return array An array of selected components for the specified subgroup.
 */
function configurator_get_subgroup_selected_components( $subgroup_uid, $active_components ) {
	$selected_components = array();

	foreach ( $active_components as $component ) {
		$filtered_components = wp_list_filter(
			$component,
			array(
				'uid'  => $subgroup_uid,
				'type' => 'sub_group',
			)
		);
		if ( ! empty( $filtered_components ) ) {
			$selected_components = array_merge( $selected_components, $component );
		}
	}

	return $selected_components;
}

/**
 * Validates that at least one image child component is selected.
 *
 * @param array $component The component containing children.
 * @param array $user_selected_components User-selected component data.
 * @return array An array containing a single error message keyed by component UID, or an empty array.
 */
function configurator_validate_required_image_children( $component, $user_selected_components ) {
	$selected_uids = configurator_get_uids_from_array( $user_selected_components );

	foreach ( $component['children'] as $child ) {
		if ( 'image' === $child['type'] && in_array( $child['uid'], $selected_uids, true ) ) {
			// At least one image selected
			return array();
		}
	}

	// No image child selected — return a single error message keyed by component UID
	return array(
		$component['uid'] => sprintf(
			__( "Please select at least one image child component for '%s'.", 'your-text-domain' ),
			$component['name']
		),
	);
}
