<?php
namespace WPC;

use WPC_Utils;

final class Add_To_Cart_Validation {
	/**
	 * The version of the plugin.
	 *
	 * @var string
	 */
	public $version = '0.1.0';

	/**
	 * The single instance of the class.
	 */
	protected static $_instance = null;

	/**
	 * Returns the main instance of the Add_To_Cart_Validation class.
	 *
	 * @static
	 * @return Add_To_Cart_Validation The main instance.
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	/**
	 * Initializes the Add_To_Cart_Validation class.
	 *
	 * This constructor defines constants, includes necessary files, and triggers the
	 * 'wpc_add_to_cart_validation_loaded' action hook.
	 */
	public function __construct() {
		$this->define_constants();
		$this->includes();
		$this->init_hooks();

		do_action( 'wpc_add_to_cart_validation_loaded' );
	}

	/**
	 * Define Constants
	 */
	private function define_constants() {
		$this->define( 'WPC_ADD_TO_CART_VALIDATION_DEV_MODE', false );
		$this->define( 'WPC_ADD_TO_CART_VALIDATION_ABSPATH', dirname( WPC_ADD_TO_CART_VALIDATION_FILE ) . '/' );
		$this->define( 'WPC_ADD_TO_CART_VALIDATION_PLUGIN_BASENAME', plugin_basename( WPC_ADD_TO_CART_VALIDATION_FILE ) );
		$this->define( 'WPC_ADD_TO_CART_VALIDATION_VERSION', $this->version );
		$this->define( 'WPC_ADD_TO_CART_VALIDATION_PLUGIN_URL', untrailingslashit( plugins_url( '/', WPC_ADD_TO_CART_VALIDATION_FILE ) ) );
		$this->define( 'WPC_ADD_TO_CART_VALIDATION_ASSETS_URL', WPC_ADD_TO_CART_VALIDATION_PLUGIN_URL . '/assets/' );
	}

	/**
	 * Includes necessary files for the functionality.
	 */
	private function includes() {
		require_once WPC_ADD_TO_CART_VALIDATION_ABSPATH . 'includes/functions.php';
	}

	/**
	 * Hook into actions and filters.
	 */
	private function init_hooks() {
		add_action( 'plugins_loaded', array( $this, 'plugins_loaded' ) );
		add_filter( 'wpc_allow_add_to_cart', array( $this, 'validate_add_to_cart' ), 10, 3 );
	}

	/**
	 * Handles the plugins_loaded action.
	 *
	 * This method checks if the WP Configurator Pro plugin is installed and activated.
	 * If the plugin is not available, it adds an admin notice to inform the user.
	 * Additionally, it includes utility and frontend configuration classes required
	 * for the plugin's functionality.
	 *
	 * @return void
	 */
	public function plugins_loaded() {
		// Check if WP Configurator Pro installed and activated.
		if ( ! class_exists( 'WPC' ) ) {
			add_action( 'admin_notices', array( $this, 'missing_wpc_notice' ) );
			return;
		}
	}

	/**
	 * WP Configurator Pro fallback notice.
	 *
	 * @since 0.1.0
	 */
	public function missing_wpc_notice() {
		$install_url = admin_url( 'plugins.php' );

		$admin_notice_content = sprintf(
			// translators: 1$ and 2$ are <strong> tags, 3$-4$ are WP Configurator Pro site link tags, 5$-6$ are install link tags.
			esc_html__(
				'%1$sAdd to Cart Validation for WP Configurator Pro%2$s requires the %3$sWP Configurator Pro%4$s plugin to function properly. Please %5$sinstall and activate WP Configurator Pro%6$s to continue.',
				'print-export-for-wp-configurator-pro'
			),
			'<strong>',
			'</strong>',
			'<a href="https://wpconfigurator.com/" target="_blank">',
			'</a>',
			'<a href="' . esc_url( $install_url ) . '">',
			'</a>'
		);

		echo '<div class="notice notice-error">';
		echo '<p>' . wp_kses_post( $admin_notice_content ) . '</p>';
		echo '</div>';
	}

	/**
	 * Get the plugin url.
	 *
	 * @return string
	 */
	public function plugin_url() {
		return untrailingslashit( plugins_url( '/', WPC_ADD_TO_CART_VALIDATION_FILE ) );
	}

	/**
	 * Get the plugin path.
	 *
	 * @return string
	 */
	public function plugin_path() {
		return untrailingslashit( plugin_dir_path( WPC_ADD_TO_CART_VALIDATION_FILE ) );
	}

	/**
	 * Define constant if not already set.
	 *
	 * @param string      $name  Constant name.
	 * @param string|bool $value Constant value.
	 */
	private function define( $name, $value ) {
		if ( ! defined( $name ) ) {
			define( $name, $value );
		}
	}

	/**
	 * Filter the 'wpc_allow_add_to_cart' hook to validate components before adding a product to the cart.
	 *
	 * This function checks if the product has a configuration ID and associated components.
	 * It validates the components against the selected cart data, and if there are validation errors,
	 * the product is not allowed to be added to the cart, and an error notice is shown.
	 *
	 * @param bool  $allow      Whether the product is allowed to be added to the cart.
	 * @param int   $product_id The ID of the product being added to the cart.
	 * @param array $cart_data  The cart data, which includes 'tree_set' that contains the selected components.
	 *
	 * @return bool The modified $allow value (false if there are validation errors, true otherwise).
	 */
	public function validate_add_to_cart( $allow, $product_id, $cart_data ) {
		// Get config ID for the product
		$config_id = absint( WPC_Utils::get_meta_value( $product_id, '_wpc_config_id', false ) );

		// If there's no config ID, allow adding the product to the cart
		if ( ! $config_id ) {
			return $allow;
		}

		// Retrieve components and filter them recursively
		$components          = WPC_Utils::get_meta_value( $config_id, '_wpc_components', array() );
		$filtered_components = array_map( 'array_filter', WPC_Utils::array_filter_recursive( $components ) );

		// Validate components against the cart data
		if ( empty( $cart_data['tree_set'] ) ) {
			return $allow; // If no tree set in cart data, return the original $allow value
		}

		$validation_errors = configurator_validate_components( $filtered_components, $cart_data['tree_set'] );

		// If validation errors are found, disallow and show the first error
		if ( ! empty( $validation_errors ) ) {
			// $allow       = false;
			// $first_error = reset( $validation_errors );
			// wc_add_notice( $first_error, 'error' );

			foreach ( $validation_errors as $error ) {
				wc_add_notice( $error, 'error' );
			}

			$allow = false;
		}

		return $allow;
	}
}
